from fastapi import FastAPI, Request, Depends
from fastapi.middleware.cors import CORSMiddleware
from fastapi.middleware.trustedhost import TrustedHostMiddleware
from fastapi.responses import JSONResponse
from fastapi.exceptions import RequestValidationError
from contextlib import asynccontextmanager
import logging
import time

from app.database import init_db
from app.api import auth
from app.core.config import settings
from app.core.deps import get_current_user

# تنظیمات لاگینگ
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


@asynccontextmanager
async def lifespan(app: FastAPI):
    """
    مدیریت چرخه حیات برنامه
    """
    # راه‌اندازی
    logger.info("🚀 راه‌اندازی دستیار هوش مصنوعی مستقل...")
    
    # ایجاد جداول پایگاه داده
    try:
        init_db()
        logger.info("✅ پایگاه داده آماده است")
    except Exception as e:
        logger.error(f"❌ خطا در راه‌اندازی پایگاه داده: {e}")
        raise
    
    yield
    
    # خاموش‌سازی
    logger.info("🛑 خاموش‌سازی برنامه...")


# ایجاد برنامه FastAPI
app = FastAPI(
    title=settings.PROJECT_NAME,
    version=settings.VERSION,
    openapi_url=f"/api/openapi.json",
    docs_url="/api/docs",
    redoc_url="/api/redoc",
    lifespan=lifespan
)

# Middleware ها
app.add_middleware(
    CORSMiddleware,
    allow_origins=settings.BACKEND_CORS_ORIGINS,
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

app.add_middleware(
    TrustedHostMiddleware,
    allowed_hosts=["*"] if settings.DEBUG else [settings.DOMAIN]
)


# Middleware زمان‌سنج
@app.middleware("http")
async def add_process_time_header(request: Request, call_next):
    start_time = time.time()
    response = await call_next(request)
    process_time = time.time() - start_time
    response.headers["X-Process-Time"] = str(process_time)
    return response


# هندلر خطاها
@app.exception_handler(RequestValidationError)
async def validation_exception_handler(request: Request, exc: RequestValidationError):
    return JSONResponse(
        status_code=422,
        content={
            "detail": "خطا در اعتبارسنجی داده‌ها",
            "errors": exc.errors()
        },
    )


@app.exception_handler(Exception)
async def general_exception_handler(request: Request, exc: Exception):
    logger.error(f"خطای سرور: {exc}", exc_info=True)
    return JSONResponse(
        status_code=500,
        content={
            "detail": "خطای داخلی سرور",
            "message": "لطفاً بعداً تلاش کنید"
        },
    )


# روت‌های پایه
@app.get("/")
async def root():
    return {
        "message": "به دستیار هوش مصنوعی مستقل خوش آمدید",
        "version": settings.VERSION,
        "docs": "/api/docs"
    }


@app.get("/api/health")
async def health_check():
    return {
        "status": "healthy",
        "timestamp": time.time(),
        "service": settings.PROJECT_NAME
    }


@app.get("/api/protected-test")
async def protected_test(current_user=Depends(get_current_user)):
    return {
        "message": "دسترسی موفق",
        "user": current_user.username,
        "is_admin": current_user.is_superuser
    }


# اضافه کردن روت‌های auth
app.include_router(auth.router, prefix="/api/auth", tags=["احراز هویت"])


if __name__ == "__main__":
    import uvicorn
    uvicorn.run(
        "main:app",
        host="0.0.0.0",
        port=8000,
        reload=settings.DEBUG,
        log_level="info"
    )
# بعد از imports موجود، اینها را اضافه کنید
from app.api import auth, tasks, files, queue, admin
from app.core.config import settings
from app.celery_app import celery_app

# بعد از ایجاد app، middleware‌ها را اضافه کنید
app.add_middleware(
    CORSMiddleware,
    allow_origins=settings.BACKEND_CORS_ORIGINS,
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# بعد از روت‌های پایه، روت‌های جدید را اضافه کنید
app.include_router(auth.router, prefix="/api/auth", tags=["احراز هویت"])
app.include_router(tasks.router, prefix="/api/tasks", tags=["تسک‌ها"])
app.include_router(files.router, prefix="/api/files", tags=["فایل‌ها"])
app.include_router(queue.router, prefix="/api/queue", tags=["صف"])
app.include_router(admin.router, prefix="/api/admin", tags=["مدیریت"])

# روت سلامت Celery
@app.get("/api/celery-health")
async def celery_health():
    try:
        # تست اتصال به Celery
        result = celery_app.control.inspect().active()
        return {
            "status": "healthy",
            "celery_connected": bool(result),
            "workers": list(result.keys()) if result else []
        }
    except Exception as e:
        return {
            "status": "unhealthy",
            "error": str(e)
        }
