from celery import shared_task
from sqlalchemy.orm import Session
from app.database import SessionLocal
from app.crud import task_crud, file_crud
from app.models import Task, TaskQueue
from app.schemas import RiskLevel
import logging
import time
import json
from datetime import datetime

logger = logging.getLogger(__name__)


@shared_task(bind=True, max_retries=3)
def process_task(self, task_id: str):
    """
    پردازش تسک در صف
    """
    db = SessionLocal()
    try:
        # دریافت تسک
        task = task_crud.get_by_id(db, task_id)
        if not task:
            logger.error(f"تسک {task_id} یافت نشد")
            return
        
        # علامت‌گذاری به عنوان در حال پردازش
        task_crud.mark_task_as_processing(db, task_id, self.request.id)
        
        # تحلیل هدف و ریسک
        risk_assessment = analyze_task_risk(task)
        task.risk_level = risk_assessment.level
        db.commit()
        
        # ثبت لاگ تحلیل
        task_crud.add_log(db, task_id, "info", "تحلیل ریسک انجام شد", {
            "risk_level": risk_assessment.level,
            "description": risk_assessment.description
        })
        
        # بررسی نیاز به تأیید
        if risk_assessment.required_confirmation and task.requires_confirmation:
            task_crud.add_log(db, task_id, "warning", "این تسک نیاز به تأیید کاربر دارد", {
                "confirmation_code": risk_assessment.confirmation_code
            })
            # در اینجا باید منتظر تأیید کاربر بمانیم
            # فعلاً ادامه می‌دهیم برای تست
        
        # اجرای تسک بر اساس نوع
        if task.task_type == "desktop":
            result = execute_desktop_task(task, db)
        elif task.task_type == "web":
            result = execute_web_task(task, db)
        elif task.task_type == "analysis":
            result = execute_analysis_task(task, db)
        else:
            result = execute_general_task(task, db)
        
        # ذخیره نتیجه
        if result.get("success"):
            task_crud.mark_task_as_completed(db, task_id, result)
            logger.info(f"تسک {task_id} با موفقیت تکمیل شد")
        else:
            task_crud.mark_task_as_failed(db, task_id, result.get("error", "خطای نامشخص"))
            logger.error(f"تسک {task_id} ناموفق بود: {result.get('error')}")
        
    except Exception as e:
        logger.error(f"خطا در پردازش تسک {task_id}: {e}")
        
        # علامت‌گذاری به عنوان ناموفق
        task_crud.mark_task_as_failed(db, task_id, str(e))
        
        # تلاش مجدد
        self.retry(countdown=60, exc=e)
    
    finally:
        db.close()


def analyze_task_risk(task: Task):
    """تحلیل ریسک تسک"""
    from app.core.security import generate_confirmation_code
    
    goal_lower = task.goal.lower()
    
    # تحلیل بر اساس کلمات کلیدی
    high_risk_keywords = [
        "حذف", "پاک", "فرمت", "سیستم", "رمز", "لاگین",
        "جعل", "تقلب", "هک", "نفوذ", "دسترسی کامل"
    ]
    
    medium_risk_keywords = [
        "دانلود", "آپلود", "اجرای کد", "اسکرپ", "اطلاعات",
        "سیستم عامل", "برنامه", "نصب"
    ]
    
    # تشخیص سطح ریسک
    risk_level = RiskLevel.LOW
    description = "ریسک پایین - اقدامات عادی"
    
    for keyword in high_risk_keywords:
        if keyword in goal_lower:
            risk_level = RiskLevel.HIGH
            description = "ریسک بالا - اقدامات حساس"
            break
    
    if risk_level == RiskLevel.LOW:
        for keyword in medium_risk_keywords:
            if keyword in goal_lower:
                risk_level = RiskLevel.MEDIUM
                description = "ریسک متوسط - نیاز به توجه"
                break
    
    # بررسی موارد خاص
    if any(x in goal_lower for x in ["همه چیز", "کنترل کامل", "دسترسی کامل"]):
        risk_level = RiskLevel.EXTREME
        description = "ریسک بسیار بالا - کنترل کامل سیستم"
    
    return {
        "level": risk_level,
        "description": description,
        "legal_warnings": get_legal_warnings(risk_level),
        "potential_damage": get_potential_damage(risk_level),
        "required_confirmation": risk_level in [RiskLevel.HIGH, RiskLevel.EXTREME],
        "confirmation_code": generate_confirmation_code() if risk_level in [RiskLevel.HIGH, RiskLevel.EXTREME] else None
    }


def get_legal_warnings(risk_level: RiskLevel) -> list:
    """هشدارهای قانونی بر اساس سطح ریسک"""
    warnings = {
        RiskLevel.LOW: [],
        RiskLevel.MEDIUM: [
            "این عمل ممکن است بر عملکرد سیستم تأثیر بگذارد",
            "اطلاعات شخصی ممکن است در معرض دید قرار گیرد"
        ],
        RiskLevel.HIGH: [
            "ممکن است قوانین حریم خصوصی را نقض کند",
            "امکان آسیب به سیستم وجود دارد",
            "ممکن است منجر به از دست دادن داده شود"
        ],
        RiskLevel.EXTREME: [
            "ممکن است مصداق جرم رایانه‌ای باشد",
            "امکان پیگرد قانونی وجود دارد",
            "اعتماد دیگران ممکن است از بین برود",
            "خطر امنیتی جدی برای سیستم"
        ]
    }
    return warnings.get(risk_level, [])


def get_potential_damage(risk_level: RiskLevel) -> str:
    """آسیب‌های بالقوه"""
    damages = {
        RiskLevel.LOW: "هیچ آسیب جدی",
        RiskLevel.MEDIUM: "آسیب جزئی به سیستم یا داده‌ها",
        RiskLevel.HIGH: "آسیب به سیستم، از دست دادن داده‌ها، نقض حریم خصوصی",
        RiskLevel.EXTREME: "آسیب جبران‌ناپذیر، مشکلات قانونی، از دست دادن اعتماد"
    }
    return damages.get(risk_level, "نامشخص")


def execute_desktop_task(task: Task, db: Session):
    """اجرای تسک کنترل دسکتاپ"""
    from app.tasks.desktop_control import DesktopController
    
    controller = DesktopController()
    result = controller.execute_task(task.goal)
    
    # ذخیره اسکرین‌شات‌ها اگر وجود دارند
    if result.get("screenshots"):
        for screenshot_data in result["screenshots"]:
            # در اینجا باید فایل ذخیره شود
            pass
    
    return result


def execute_web_task(task: Task, db: Session):
    """اجرای تسک وب"""
    from app.tasks.web_scraping import WebScraper
    
    scraper = WebScraper()
    result = scraper.scrape_website(task.goal)
    
    return result


def execute_analysis_task(task: Task, db: Session):
    """اجرای تسک تحلیل"""
    # تحلیل متن با DeepSeek
    result = {
        "success": True,
        "analysis": "تحلیل اولیه انجام شد",
        "recommendations": [],
        "data": {}
    }
    
    return result


def execute_general_task(task: Task, db: Session):
    """اجرای تسک عمومی"""
    # شبیه‌سازی پردازش
    time.sleep(2)
    
    return {
        "success": True,
        "message": "تسک با موفقیت اجرا شد",
        "execution_time": 2.0,
        "result": "نتیجه تستی"
    }


@shared_task
def queue_task_processor():
    """پردازشگر صف - تسک‌های در انتظار را پردازش می‌کند"""
    db = SessionLocal()
    try:
        # دریافت تسک بعدی در صف
        next_task = task_crud.get_next_pending_task(db)
        
        if not next_task:
            logger.debug("هیچ تسکی در صف نیست")
            return
        
        logger.info(f"پردازش تسک {next_task.task_id} از صف")
        
        # شروع پردازش
        process_task.delay(next_task.task_id)
        
    except Exception as e:
        logger.error(f"خطا در پردازشگر صف: {e}")
    finally:
        db.close()
