from pydantic import BaseModel, Field, validator
from typing import Optional, List, Dict, Any
from datetime import datetime
from enum import Enum


class TaskStatus(str, Enum):
    PENDING = "pending"
    QUEUED = "queued"
    PROCESSING = "processing"
    COMPLETED = "completed"
    FAILED = "failed"
    CANCELLED = "cancelled"


class TaskType(str, Enum):
    GENERAL = "general"
    DESKTOP = "desktop"
    WEB = "web"
    ANALYSIS = "analysis"
    AUTOMATION = "automation"
    SECURITY = "security"


class RiskLevel(str, Enum):
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    EXTREME = "extreme"


# تسک
class TaskBase(BaseModel):
    title: str = Field(..., min_length=1, max_length=200, description="عنوان تسک")
    description: Optional[str] = Field(None, description="توضیحات تسک")
    goal: str = Field(..., min_length=10, description="هدف تسک")
    task_type: TaskType = Field(TaskType.GENERAL, description="نوع تسک")
    priority: int = Field(5, ge=1, le=10, description="اولویت (1-10)")
    requires_confirmation: bool = Field(True, description="نیاز به تأیید دارد")
    max_execution_time: Optional[int] = Field(None, description="حداکثر زمان اجرا (ثانیه)")


class TaskCreate(TaskBase):
    pass


class TaskUpdate(BaseModel):
    title: Optional[str] = None
    description: Optional[str] = None
    status: Optional[TaskStatus] = None
    progress: Optional[int] = Field(None, ge=0, le=100)
    result_data: Optional[Dict[str, Any]] = None
    error_message: Optional[str] = None


class TaskInDB(TaskBase):
    id: str
    user_id: int
    status: TaskStatus
    progress: int
    risk_level: Optional[RiskLevel]
    result_data: Optional[Dict[str, Any]]
    result_files: Optional[List[str]]
    error_message: Optional[str]
    execution_time: Optional[float]
    created_at: datetime
    queued_at: Optional[datetime]
    started_at: Optional[datetime]
    completed_at: Optional[datetime]
    
    class Config:
        from_attributes = True


# اقدام
class ActionBase(BaseModel):
    action_type: str = Field(..., description="نوع اقدام")
    description: str = Field(..., description="توضیح اقدام")
    parameters: Dict[str, Any] = Field(default_factory=dict)
    requires_confirmation: bool = Field(False, description="نیاز به تأیید دارد")
    confirmation_code: Optional[str] = Field(None, description="کد تأیید")


class ActionCreate(ActionBase):
    pass


class ActionInDB(ActionBase):
    id: int
    task_id: str
    status: str
    risk_assessment: Optional[Dict[str, Any]]
    result: Optional[Dict[str, Any]]
    confirmed: bool
    created_at: datetime
    confirmed_at: Optional[datetime]
    executed_at: Optional[datetime]
    
    class Config:
        from_attributes = True


# فایل
class FileBase(BaseModel):
    original_filename: str
    file_type: str
    is_temp: bool = True


class FileCreate(FileBase):
    pass


class FileInDB(FileBase):
    id: str
    task_id: Optional[str]
    user_id: int
    filename: str
    file_path: str
    file_size: int
    mime_type: Optional[str]
    expires_at: Optional[datetime]
    created_at: datetime
    
    class Config:
        from_attributes = True


# لاگ
class TaskLogBase(BaseModel):
    log_level: str
    message: str
    details: Optional[Dict[str, Any]] = None


class TaskLogInDB(TaskLogBase):
    id: int
    task_id: str
    created_at: datetime
    
    class Config:
        from_attributes = True


# ریسک
class RiskAssessment(BaseModel):
    level: RiskLevel
    description: str
    legal_warnings: List[str] = []
    potential_damage: str
    required_confirmation: bool
    confirmation_code: Optional[str] = None


# وضعیت صف
class QueueStatus(BaseModel):
    task_id: str
    position: int
    queue_status: str
    estimated_wait_time: Optional[int] = None
    active_workers: int
    pending_tasks: int


# آمار
class TaskStatistics(BaseModel):
    total_tasks: int
    completed_tasks: int
    failed_tasks: int
    pending_tasks: int
    average_execution_time: float
    success_rate: float
    recent_tasks: List[TaskInDB]
