from sqlalchemy.orm import Session
from typing import Optional, List, Dict, Any
from app.models import Task, Action, TaskFile, TaskLog, TaskQueue
from app.schemas import TaskCreate, TaskUpdate, ActionCreate
from datetime import datetime, timedelta
import uuid
import logging
import os
from app.core.config import settings

logger = logging.getLogger(__name__)


class TaskCRUD:
    @staticmethod
    def generate_task_id() -> str:
        """ایجاد شناسه منحصربه‌فرد برای تسک"""
        return str(uuid.uuid4())
    
    @staticmethod
    def get_by_id(db: Session, task_id: str) -> Optional[Task]:
        return db.query(Task).filter(Task.id == task_id).first()
    
    @staticmethod
    def get_by_user(db: Session, user_id: int, skip: int = 0, limit: int = 100) -> List[Task]:
        return db.query(Task).filter(Task.user_id == user_id).order_by(Task.created_at.desc()).offset(skip).limit(limit).all()
    
    @staticmethod
    def get_active_tasks(db: Session, user_id: int) -> List[Task]:
        return db.query(Task).filter(
            Task.user_id == user_id,
            Task.status.in_(["pending", "queued", "processing"])
        ).all()
    
    @staticmethod
    def create(db: Session, user_id: int, obj_in: TaskCreate) -> Task:
        task_id = TaskCRUD.generate_task_id()
        
        db_task = Task(
            id=task_id,
            user_id=user_id,
            title=obj_in.title,
            description=obj_in.description,
            goal=obj_in.goal,
            task_type=obj_in.task_type,
            status="pending",
            progress=0,
            priority=obj_in.priority if hasattr(obj_in, 'priority') else 5,
            requires_confirmation=obj_in.requires_confirmation if hasattr(obj_in, 'requires_confirmation') else True,
            max_execution_time=obj_in.max_execution_time if hasattr(obj_in, 'max_execution_time') else settings.MAX_TASK_DURATION_SECONDS
        )
        
        db.add(db_task)
        
        # اضافه کردن به صف
        from app.models import TaskQueue
        queue_entry = TaskQueue(
            task_id=task_id,
            user_id=user_id,
            queue_status="pending",
            position=TaskCRUD.get_next_queue_position(db, user_id)
        )
        db.add(queue_entry)
        
        # ثبت لاگ
        TaskCRUD.add_log(db, task_id, "info", "تسک ایجاد شد", {
            "title": obj_in.title,
            "task_type": obj_in.task_type
        })
        
        # ثبت در لاگ سیستمی
        from app.models import AuditLog
        audit_log = AuditLog(
            user_id=user_id,
            event_type="task_create",
            event_details={
                "task_id": task_id,
                "title": obj_in.title,
                "task_type": obj_in.task_type
            },
            severity="info"
        )
        db.add(audit_log)
        
        db.commit()
        db.refresh(db_task)
        
        logger.info(f"تسک جدید ایجاد شد: {task_id} توسط کاربر {user_id}")
        
        return db_task
    
    @staticmethod
    def get_next_queue_position(db: Session, user_id: int) -> int:
        """دریافت موقعیت بعدی در صف کاربر"""
        last_position = db.query(TaskQueue.position).filter(
            TaskQueue.user_id == user_id,
            TaskQueue.queue_status.in_(["pending", "processing"])
        ).order_by(TaskQueue.position.desc()).first()
        
        return (last_position[0] + 1) if last_position else 1
    
    @staticmethod
    def update(db: Session, db_task: Task, obj_in: TaskUpdate) -> Task:
        update_data = obj_in.dict(exclude_unset=True)
        
        for field, value in update_data.items():
            setattr(db_task, field, value)
        
        # ثبت لاگ تغییرات
        TaskCRUD.add_log(db, db_task.id, "info", "تسک به‌روزرسانی شد", {
            "updated_fields": list(update_data.keys())
        })
        
        db.commit()
        db.refresh(db_task)
        
        return db_task
    
    @staticmethod
    def delete(db: Session, task_id: str) -> bool:
        db_task = TaskCRUD.get_by_id(db, task_id)
        if not db_task:
            return False
        
        # حذف فایل‌های مرتبط
        for file in db_task.files:
            try:
                if os.path.exists(file.file_path):
                    os.remove(file.file_path)
            except:
                pass
        
        # ثبت در لاگ سیستمی
        from app.models import AuditLog
        audit_log = AuditLog(
            user_id=db_task.user_id,
            event_type="task_delete",
            event_details={
                "task_id": task_id,
                "title": db_task.title
            },
            severity="warning"
        )
        db.add(audit_log)
        
        db.delete(db_task)
        db.commit()
        
        logger.warning(f"تسک حذف شد: {task_id}")
        return True
    
    @staticmethod
    def add_action(db: Session, task_id: str, action_in: ActionCreate) -> Action:
        # بررسی وجود تسک
        task = TaskCRUD.get_by_id(db, task_id)
        if not task:
            raise ValueError("تسک یافت نشد")
        
        db_action = Action(
            task_id=task_id,
            action_type=action_in.action_type,
            description=action_in.description,
            parameters=action_in.parameters,
            status="pending",
            requires_confirmation=action_in.requires_confirmation,
            confirmation_code=action_in.confirmation_code if hasattr(action_in, 'confirmation_code') else None
        )
        
        db.add(db_action)
        
        # ثبت لاگ
        TaskCRUD.add_log(db, task_id, "info", "اقدام جدید افزوده شد", {
            "action_type": action_in.action_type,
            "description": action_in.description
        })
        
        db.commit()
        db.refresh(db_action)
        
        return db_action
    
    @staticmethod
    def add_log(db: Session, task_id: str, log_level: str, message: str, details: Dict = None):
        """افزودن لاگ به تسک"""
        log = TaskLog(
            task_id=task_id,
            log_level=log_level,
            message=message,
            details=details or {}
        )
        db.add(log)
        db.commit()
    
    @staticmethod
    def get_task_logs(db: Session, task_id: str, limit: int = 100) -> List[TaskLog]:
        """دریافت لاگ‌های تسک"""
        return db.query(TaskLog).filter(
            TaskLog.task_id == task_id
        ).order_by(TaskLog.created_at.desc()).limit(limit).all()
    
    @staticmethod
    def update_task_status(db: Session, task_id: str, status: str, progress: int = None, error: str = None):
        """به‌روزرسانی وضعیت تسک"""
        task = TaskCRUD.get_by_id(db, task_id)
        if not task:
            return
        
        old_status = task.status
        task.status = status
        
        if progress is not None:
            task.progress = progress
        
        if error:
            task.error_message = error
        
        # ثبت زمان‌ها
        if status == "queued":
            task.queued_at = datetime.utcnow()
        elif status == "processing":
            task.started_at = datetime.utcnow()
        elif status in ["completed", "failed", "cancelled"]:
            task.completed_at = datetime.utcnow()
            if task.started_at:
                task.execution_time = (datetime.utcnow() - task.started_at).total_seconds()
        
        # ثبت لاگ تغییر وضعیت
        TaskCRUD.add_log(db, task_id, "info", f"وضعیت تسک تغییر کرد: {old_status} → {status}")
        
        db.commit()
        db.refresh(task)
        return task
    
    @staticmethod
    def get_next_pending_task(db: Session) -> Optional[TaskQueue]:
        """دریافت تسک بعدی در صف"""
        return db.query(TaskQueue).filter(
            TaskQueue.queue_status == "pending"
        ).order_by(
            TaskQueue.position.asc()
        ).first()
    
    @staticmethod
    def mark_task_as_processing(db: Session, task_id: str, worker_id: str) -> bool:
        """علامت‌گذاری تسک به عنوان در حال پردازش"""
        queue_entry = db.query(TaskQueue).filter(
            TaskQueue.task_id == task_id
        ).first()
        
        if not queue_entry:
            return False
        
        queue_entry.queue_status = "processing"
        queue_entry.started_processing_at = datetime.utcnow()
        queue_entry.worker_id = worker_id
        
        # به‌روزرسانی وضعیت تسک
        TaskCRUD.update_task_status(db, task_id, "processing")
        
        db.commit()
        return True
    
    @staticmethod
    def mark_task_as_completed(db: Session, task_id: str, result_data: Dict = None) -> bool:
        """علامت‌گذاری تسک به عنوان تکمیل شده"""
        queue_entry = db.query(TaskQueue).filter(
            TaskQueue.task_id == task_id
        ).first()
        
        if not queue_entry:
            return False
        
        queue_entry.queue_status = "completed"
        queue_entry.completed_at = datetime.utcnow()
        
        # به‌روزرسانی وضعیت تسک
        task = TaskCRUD.update_task_status(db, task_id, "completed", progress=100)
        if task and result_data:
            task.result_data = result_data
        
        db.commit()
        return True
    
    @staticmethod
    def mark_task_as_failed(db: Session, task_id: str, error_message: str) -> bool:
        """علامت‌گذاری تسک به عنوان ناموفق"""
        queue_entry = db.query(TaskQueue).filter(
            TaskQueue.task_id == task_id
        ).first()
        
        if not queue_entry:
            return False
        
        queue_entry.queue_status = "failed"
        
        # به‌روزرسانی وضعیت تسک
        TaskCRUD.update_task_status(db, task_id, "failed", error=error_message)
        
        db.commit()
        return True


task_crud = TaskCRUD()


class FileCRUD:
    """مدیریت فایل‌های تسک"""
    
    @staticmethod
    def save_uploaded_file(file, user_id: int, task_id: str = None, file_type: str = "upload") -> Optional[TaskFile]:
        """ذخیره فایل آپلود شده"""
        from app.database import SessionLocal
        db = SessionLocal()
        
        try:
            # ایجاد شناسه فایل
            file_id = str(uuid.uuid4())
            
            # ایجاد نام فایل امن
            original_filename = file.filename
            file_ext = os.path.splitext(original_filename)[1]
            safe_filename = f"{file_id}{file_ext}"
            
            # تعیین مسیر ذخیره‌سازی
            if file_type == "result":
                save_dir = settings.RESULTS_DIR
            elif file_type == "screenshot":
                save_dir = settings.SCREENSHOTS_DIR
            else:
                save_dir = settings.UPLOAD_DIR
            
            os.makedirs(save_dir, exist_ok=True)
            file_path = os.path.join(save_dir, safe_filename)
            
            # ذخیره فایل
            file_content = file.file.read()
            with open(file_path, "wb") as f:
                f.write(file_content)
            
            # محاسبه اندازه
            file_size = len(file_content)
            
            # ایجاد رکورد در دیتابیس
            db_file = TaskFile(
                id=file_id,
                task_id=task_id,
                user_id=user_id,
                filename=safe_filename,
                original_filename=original_filename,
                file_path=file_path,
                file_size=file_size,
                mime_type=file.content_type,
                file_type=file_type,
                is_temp=True,
                expires_at=datetime.utcnow() + timedelta(hours=settings.TEMP_FILE_MAX_AGE_HOURS)
            )
            
            db.add(db_file)
            db.commit()
            db.refresh(db_file)
            
            logger.info(f"فایل ذخیره شد: {original_filename} ({file_size} بایت)")
            return db_file
            
        except Exception as e:
            logger.error(f"خطا در ذخیره فایل: {e}")
            db.rollback()
            return None
        finally:
            db.close()
    
    @staticmethod
    def get_file_by_id(db: Session, file_id: str) -> Optional[TaskFile]:
        """دریافت فایل بر اساس شناسه"""
        return db.query(TaskFile).filter(TaskFile.id == file_id).first()
    
    @staticmethod
    def get_user_files(db: Session, user_id: int, task_id: str = None) -> List[TaskFile]:
        """دریافت فایل‌های کاربر"""
        query = db.query(TaskFile).filter(TaskFile.user_id == user_id)
        
        if task_id:
            query = query.filter(TaskFile.task_id == task_id)
        
        return query.order_by(TaskFile.created_at.desc()).all()
    
    @staticmethod
    def cleanup_expired_files(db: Session) -> int:
        """پاکسازی فایل‌های منقضی شده"""
        expired_files = db.query(TaskFile).filter(
            TaskFile.is_temp == True,
            TaskFile.expires_at < datetime.utcnow()
        ).all()
        
        deleted_count = 0
        for file in expired_files:
            try:
                # حذف فایل از دیسک
                if os.path.exists(file.file_path):
                    os.remove(file.file_path)
                
                # حذف از دیتابیس
                db.delete(file)
                deleted_count += 1
                
            except Exception as e:
                logger.error(f"خطا در حذف فایل {file.file_path}: {e}")
        
        db.commit()
        
        if deleted_count > 0:
            logger.info(f"پاکسازی فایل‌ها: {deleted_count} فایل حذف شد")
        
        return deleted_count
    
    @staticmethod
    def make_file_permanent(db: Session, file_id: str) -> bool:
        """تبدیل فایل موقت به دائمی"""
        file = FileCRUD.get_file_by_id(db, file_id)
        if not file:
            return False
        
        file.is_temp = False
        file.expires_at = None
        db.commit()
        
        return True


file_crud = FileCRUD()
