from celery import Celery
from app.core.config import settings
import os

# تنظیمات Celery
celery_app = Celery(
    "ai_assistant",
    broker=settings.CELERY_BROKER_URL,
    backend=settings.CELERY_RESULT_BACKEND,
    include=[
        "app.tasks.task_processor",
        "app.tasks.file_cleanup",
        "app.tasks.desktop_control",
        "app.tasks.web_scraping",
    ]
)

# تنظیمات Celery
celery_app.conf.update(
    task_serializer="json",
    accept_content=["json"],
    result_serializer="json",
    timezone="Asia/Tehran",
    enable_utc=True,
    
    # Worker settings
    worker_max_tasks_per_child=100,
    worker_prefetch_multiplier=1,
    
    # Task settings
    task_acks_late=True,
    task_reject_on_worker_lost=True,
    task_track_started=True,
    task_time_limit=settings.MAX_TASK_DURATION_SECONDS,
    task_soft_time_limit=settings.MAX_TASK_DURATION_SECONDS - 30,
    
    # Queue settings
    task_queues={
        'task_queue': {
            'exchange': 'tasks',
            'routing_key': 'task.process',
        },
        'high_priority': {
            'exchange': 'tasks',
            'routing_key': 'task.high',
        },
        'low_priority': {
            'exchange': 'tasks',
            'routing_key': 'task.low',
        },
    },
    
    task_default_queue='task_queue',
    task_default_exchange='tasks',
    task_default_routing_key='task.process',
    
    # Beat schedule
    beat_schedule={
        'cleanup-expired-files': {
            'task': 'app.tasks.file_cleanup.cleanup_expired_files',
            'schedule': settings.CLEANUP_INTERVAL_HOURS * 3600,  # هر 2 ساعت
        },
        'update-system-stats': {
            'task': 'app.tasks.system_tasks.update_system_statistics',
            'schedule': 300,  # هر 5 دقیقه
        },
    },
)

# ایجاد پوشه‌های لازم
os.makedirs(settings.UPLOAD_DIR, exist_ok=True)
os.makedirs(settings.TEMP_DIR, exist_ok=True)
os.makedirs(settings.SCREENSHOTS_DIR, exist_ok=True)
os.makedirs(settings.RESULTS_DIR, exist_ok=True)


@celery_app.task(bind=True)
def debug_task(self):
    """تسک دیباگ"""
    print(f"Request: {self.request!r}")
    return {"status": "ok", "task_id": self.request.id}
